;;; ------------------------------------------------------------------------
;;;		 LAYER_CREATOR.LSP v1.9.8
;;;
;;;		 Copyright October 2010
;;;		 Timothy G. Spangler
;;;
;;;		 Permission to use, copy, modify, and distribute this software
;;;		 for any purpose and without fee is hereby granted, provided
;;;		 that the above copyright notice appears in all copies and
;;;		 that both that copyright notice and the limited warranty and
;;;		 restricted rights notice below appear in all supporting
;;;		 documentation.
;;;
;;;		 Layer Creator:
;;;				The Layer Creator is a dialog driven program that enables
;;;				the user to select from differant layer groups to insert
;;;				standard layers into there drawing.  If the layer exsist in
;;;				the drawing the user can have the properties updated to the insert
;;;				layers properties.
;;;
;;;		 Files include:
;;;					Layer_Creator.lsp				- The main routine
;;;					Layer_Creator.dcl				- The dialog file containing all dialog boxes
;;;					Layer_Creator.slb				- Slide Library file
;;;
;;; ---Global Variables-----------------------------------------------------
;;;
;;;			GBL:Current - Used for set the inserted layer current
;;;			GBL:Update -  Used for updating the layers in the dwg with the layers inserted
;;;
;;; ---Version 1.0----------------------------------------------------------
;;;			04-04 = Started coding Everything is as expected.
;;;			05-15 = Finished Coding Started Testing.
;;;			06-03 = Added description for Version 2005+.
;;;			06-04 = Fixed adding description - Used ActiveX Method.
;;;
;;; ---Version 1.1----------------------------------------------------------
;;;			06-07 = Fixed directory structure to allow flexible setup.
;;;
;;; ---Version 1.2----------------------------------------------------------
;;;			10-26 = Created default layer file for automatic population on open.
;;;			11-04 = Added sub to delete directory from search path when completed.
;;;
;;; ---Version 1.3----------------------------------------------------------
;;;			03-03 = Created Seperate sub to create Layers
;;;			03-03 = Added Plot/No Plot Option
;;;			03-07 = Added Export Option to export layers from drawing
;;;			03-10 = Added Select All/ Select None Options to dialog
;;;
;;; ---Version 1.4----------------------------------------------------------
;;;			03-29 = Fixed exporting layer descriptions in ACAD2004 and below
;;;			03-29 = Changed linetype file and removed excess garbage from it. Recompiled .shp file
;;;			03-29 = Added "Make Layer Current" toggle to dialog.	Will make last selected layer current
;;;			04-12 = Added Image tile to dialog for Plot/NoPlot (just for looks)
;;;			06-13 = Changed display when multiple layers are selected
;;;			06-13 = Added Version to registry
;;;			09-24 = Removed EXPORT option
;;;			09-24 = Revamp Create Layers Sub
;;;
;;; ---Version 1.5----------------------------------------------------------
;;;			07-21 = Added version check and updater
;;;			07-21 = Added LC_SET_ENV sub
;;;			07-24 = Added Named Plotstyles and supporting subs
;;;			07-26 = Added Plotyle info to dialog
;;;
;;; ---Version 1.6----------------------------------------------------------
;;;			01-16 = Renamed functions - added LC_ prefix
;;;			01-24 = Added optional linetype file check
;;;
;;; ---Version 1.7----------------------------------------------------------
;;;			04-07 = Revised LC_READ_FILE sub to check for empty lines in the .lyr file
;;;			04-15 = Added Layer Export functionality
;;;			04-18 = Revised layer selection when no layers are selected
;;;			04-22 = Added sub to check for linetype in all .lin files in the search path
;;;			04-23 = Removed dependencies for .lin and .shx files - Files removed
;;;			04-23 = Reorganized code
;;;			04-30 = Added "Update Layers" toggle to dialog box
;;;			05-01 = Added code to update layers
;;;
;;; ---Version 1.8----------------------------------------------------------
;;;			09-12 = Revised VLA-Obj variable declarations
;;;			09-12 = Added check for color number greater than 255
;;;
;;; ---Version 1.9----------------------------------------------------------
;;;			11-01 = Added CANCEL to export function so no files are creaeted
;;;			11-11 = Added alphabetic sort to the export function
;;;			11-19 = Added listbox selection check (NO MORE THAN 530 SELECTIONS)
;;;			11-28 = Added LC_CLEAN sub to reset the registry
;;;			11-28 = Revised LC_SET_PROG sub to streamline base directory location
;;;			11-28 = Revised Export Sub when no group name is entered
;;;
;;; ---Version 1.9.5--------------------------------------------------------
;;;			07-23-09 = Revised SET_PROG sub 
;;;
;;; ---Version 1.9.6--------------------------------------------------------
;;;			12-03-09 = Revised LC_INSERT_LAYER sub added check for ACAD version in description
;;;
;;; ---Version 1.9.7--------------------------------------------------------
;;;			01-03-10 = Revised LC_INSERT_LAYER sub, variable was named incorrectly causing an error
;;;
;;; ---Version 1.9.8--------------------------------------------------------
;;;			10-11-10 = Revised linewieght to show wieghts over .9 mm
;;;
;;; ---Need to be fixed-----------------------------------------------------
;;;		Check selection for properties being the same - John Pandazopoulos
;;;		Add Selection in layer group for "<SHOW ALL>" - only allows 530 selections
;;;
;;; ------------------------------------------------------------------------

;;; ------------ PROGRAM SHORTCUT
(defun C:LC (/)(C:LAYER_CREATOR))

;;; ------------ MAIN FUNCTION
(defun C:LAYER_CREATOR (/ *error* OldCmdEcho OldClayer OldExpert OldPstyleMode ActiveDoc Space ActiveLayout 
	LayersCol CurrentLayout CurrentPstyle PlotStyleTables Version BaseDirectory LayerGroupList LayerGroupPos
	LayerNameList LayerNamePos MasterList SelLayerNameList)
	
	;; Set system varialbes
	(LC_SET_ENV)
	
	;; Error Handling Routine
	(defun *error* (Msg)

		(if (not (member Msg '("Function cancelled" "quit / exit abort")))
			(princ (strcat "\n*** Program Error: " (strcase Msg) " ***"))
			(princ "\n... Program Cancelled ...")
		)
		(while (< 0 (getvar "CMDACTIVE"))
			(command)
		)
		;; Reset Env
		(LC_RESET_ENV)
		;; Quiet Exit
		(princ)
	)
	;; Set program variables
	(LC_SET_PROG)
	;; Run Layer Creator
	(LC_RUN_LAYER)
)
;;; ------------ Begin Main Routine
(defun LC_RUN_LAYER ( / End_Dialog Layer_DCL)
	
	;; Add program description to status line
	(grtext -2 (strcat "----- Layer Creator " Version " Copyright 2010 -----"))
	
	;; Start Dialog
	(setq Layer_DCL (load_dialog "LAYER_CREATOR.dcl"))
	(if (not (new_dialog "LayerCreator" Layer_DCL))
		(progn
			(alert "Layer_Creator.dcl not found\n\nPlace file in the search path \n		 then re-run the program.")
			(exit)
		)
	)
	;; Show seperators
	(LC_SHOW_SLIDE "sep1" "HSeperator.sld" -2)

	;; Set dialog for use
	(LC_INIT_DIALOG)

	;; Action_Tile Statements
	(action_tile "layer_group" "(setq LayerGroupPos (atoi $value))(LC_INIT_DIALOG)")
	(action_tile "lyr_names" "(setq LayerNamePos (atoi $value))(LC_SET_DIALOG)")
	(action_tile "current" "(setq GBL:Current $value)")
	(action_tile "update" "(setq GBL:Update $value)")
	
	(action_tile "selall" "(LC_SELECT_ALL \"lyr_names\" (length LayerNameList))(LC_SET_DIALOG)")
	(action_tile "selnone" "(LC_SELECT_NONE)")

	(action_tile "about" "(LC_ABOUT_LAYER)")
	(action_tile "cancel" "(done_dialog 0)")
	(action_tile "insert" "(if (not LayerNamePos)(alert \"No Layers Selected\")(done_dialog 1))")
	(action_tile "export" "(done_dialog 2)")

	(setq End_Dialog (start_dialog))

	;; Check exit method
	(cond
		((= End_Dialog 0)									 ; CANCEL or ESC was selected
			(unload_dialog Layer_DCL)			
			;; Reset status line
			(grtext -2 "")
			(princ "\n User Cancelled! \n Layer Creator Ended.")
			(LC_RESET_ENV)
		)
		((= End_Dialog 1)									 ; OK was selected
			(unload_dialog Layer_DCL)
			;; Reset status line
			(grtext -2 "")
			(princ "\n Processing Selected Layers......")
			(LC_CREATE_STYLE)
			(LC_INSERT_LAYER SelLayerNameList)
		)
		((= End_Dialog 2)									 ; EXPORT was selected
			(unload_dialog Layer_DCL)
			;; Reset status line
			(princ "\n Exporting Layers From Drawing......")
			(LC_LAYER_EXPORT)
		)
	)
)
;;; ------------ DIALOG SUB-ROUTINES ------------

;;; ------------ Set up dialog for initial start (Not Tiles)
(defun LC_INIT_DIALOG (/)

	;; Set layer group tile for first use
	(LC_LOAD_LISTBOX "layer_group" LayerGroupList)
	(set_tile "layer_group"	(rtos LayerGroupPos 2 0))	
	;; Create masterlist from the selected group
	(setq MasterList (LC_READ_FILE (strcat BaseDirectory "\\Layer Files\\" (nth LayerGroupPos (LC_GET_LAYER (strcat BaseDirectory "\\Layer Files") "lyr"))".lyr")))
	;; Create list of layers and populate listbox
	(setq LayerNameList (LC_CREATE_LAYER_NAMES MasterList))
	;; Set LayerNamePos variable to first group in list
	(setq LayerNamePos 0)	
	;; Set layer name tile for first use
	(LC_LOAD_LISTBOX "lyr_names" LayerNameList)
	(set_tile "lyr_names"	(rtos LayerNamePos 2 0))	
	;; Set current toggle to value
	(set_tile "current" GBL:Current)
	;; Set update toggle to value
	(set_tile "update" GBL:Update)
	;; Set toggle to value
	(set_tile "pstyletbl" CurrentPstyle)		
	;; Set layer property tiles
	(LC_SET_DIALOG)	
	;; Dim copyright
	(mode_tile "copyright" 1)	
)
;;; ------------ Set dialog tiles after layer group is selected
(defun LC_SET_DIALOG ( / LayerColor Width Hieght )

	;; Get the items currently selected in the listbox
	(setq SelLayerNameList (get_tile "lyr_names"))
	;; Check for number selected
	(cond
		;; Equal to 0
		((= 0 (length (LC_STRING_TO_LIST SelLayerNameList " ")))			
			;; Set toggle to value
			(mode_tile "current" 0)
			(set_tile "current" GBL:Current)			
			;; Set layer name tile
			(set_tile "layer_name" (nth 0 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";")))
			;; Set layer color image tile
			(setq LayerColor (nth 4 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";")))
			(setq Width (dimx_tile "color"))
			(setq Hieght (dimx_tile "color"))
			(start_image "color")
			(fill_image 0 0 Width Hieght (atoi LayerColor))
			(end_image)
			;; Set layer color number tile
			(set_tile "colorno" LayerColor)
			;; Set linetype tile with line type number
			(set_tile "linetype" (nth 2 (LC_STRING_TO_LIST(cdr (nth LayerNamePos MasterList)) ";")))
			
			(setq t1 (nth 3 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";")))
			;; Set lineweight tile with lineweight thickness
			(if (= "-3" (nth 3 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";")))
				(set_tile "linewieght" "Default")
				;; Check that the number is less than 100
				(if (> (atoi (nth 3 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";"))) 99)
					;; Number is greater than 100, break it up
					(set_tile "linewieght" 
						(strcat
							(substr (nth 3 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";"))1 1)
							"."
							(substr (nth 3 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";"))2)
						" mm")
					)
					;; Number is less than 100
					(set_tile "linewieght" (strcat "0."(nth 3 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";"))" mm"))
				)
			)
			;; Set plot tile with plot status
			(if(= (nth 5 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";")) "0")
				(LC_SHOW_SLIDE "plot" "NPlot.sld" -2)
				(LC_SHOW_SLIDE "plot" "Plot.sld" -2)
			)			
			;; Set description tile with description of layer
			(set_tile "description" (nth 1 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";")))			
			;; Set plotsyle tile with plotstyle of layer
			(if (= (getvar "PSTYLEMODE") 0)
				(progn
					(set_tile "plotstyle" (nth 7 (LC_STRING_TO_LIST (cdr (nth LayerNamePos MasterList)) ";")))
					(mode_tile "plotstyle" 0)
				)
				(progn
					(set_tile "plotstyle" LayerColor)
					(mode_tile "plotstyle" 1)
				)
			)
		)		
		;; Greater than 0
		((< 0 (length (LC_STRING_TO_LIST SelLayerNameList " ")))			
			;; Set toggle to value
			(mode_tile "current" 0)
			(set_tile "current" GBL:Current)						
			;; Set layer name tile
			(set_tile "layer_name" (strcat (rtos (1+ (length (LC_STRING_TO_LIST SelLayerNameList " ")))2 0) " - LAYER(S) SELECTED"))			
			;; Set layer color image tile
			(setq Width (dimx_tile "color"))
			(setq Hieght (dimx_tile "color"))
			(start_image "color")
			(fill_image 0 0 Width Hieght -15)
			(end_image)			
			;; Set layer color number tile
			(set_tile "colorno" "Varies")			
			;; Set linetype tile with line type number
			(set_tile "linetype" "Varies")			
			;; Set lineweight tile with lineweight thickness
			(set_tile "linewieght" "Varies")			
			;; Set plot tile with plot status
			(LC_SHOW_SLIDE "plot" "VPlot.sld" -2)			
			;; Set description tile with description of layer
			(set_tile "description" "Varies")			
			;; Set plotsyle tile with plotstyle of layer
			(if (= (getvar "PSTYLEMODE") 0)
				(progn
					(set_tile "plotstyle" "Varies")
					(mode_tile "plotstyle" 0)
				)
				(progn
					(set_tile "plotstyle" "Varies")
					(mode_tile "plotstyle" 1)
				)
			)
		)
	)	
	;; Dialog setting funtions for plotstyles
	(if (= (getvar "PSTYLEMODE") 0)
		(progn
			(mode_tile "ctb" 1)
			(mode_tile "stb" 0)
			(set_tile "stb" "1")
			(set_tile "pstylemode" "0 - Named")			
		)
		(progn
			(mode_tile "stb" 1)
			(mode_tile "ctb" 0)
			(set_tile "ctb" "1")
			(set_tile "pstylemode" "1 - Color")			
		)
	)
)
;;; ------------ SUB TO POPULATE A LISTBOX
(defun LC_LOAD_LISTBOX (Listbox NewList /)

	;; Clear all contents in Listbox before loading
	(start_list Listbox 3)
	;; Add the list to the edit box
	(mapcar 'add_list NewList)
	(end_list)
)
;;; ------------ SELECT NO ITEMS IN LISTBOX
(defun LC_SELECT_NONE (/ Width Hieght)

	;; Set the layer name position to nil
	(setq LayerNamePos nil)
	;; Deselect any layers in the list
	(set_tile "lyr_names" "")
	;; Set layer name tile
	(set_tile "layer_name" "NO LAYER(S) SELECTED")
	;; Set layer color image tile
	(setq Width (dimx_tile "color"))
	(setq Hieght (dimx_tile "color"))
	(start_image "color")
	(fill_image 0 0 Width Hieght -15)
	(end_image)
	;; Set layer color number tile
	(set_tile "colorno" "")
	;; Set linetype tile with line type number
	(set_tile "linetype" "")
	;; Set lineweight tile with lineweight thickness
	(set_tile "linewieght" "")
	;; Set plot tile with plot status
	(LC_SHOW_SLIDE "plot" "VPlot.sld" -2)	
	;; Set description tile with description of layer
	(set_tile "description" "")			
	;; Set plotsyle tile with plotstyle of layer
	(set_tile "plotstyle" "")
	;; Set toggle to current layer
	(set_tile "current" "0")
	(mode_tile "current" 1)
)
;;; ------------ SELECT ALL ITEMS IN LISTBOX
(defun LC_SELECT_ALL (Key Count / StartCount Value Count)

	;; Check for maximum selection
	(if (> Count 530)
		(progn
			(alert "Listbox Selecteion Exeeded \n Only 530 items will be selected")
			(setq Count 530)
		)
	)
	(setq StartCount 0)
	(setq Value "0")
	(repeat (1- Count)
		(setq Value (strcat Value (itoa (setq StartCount (1+ StartCount))) " "))
	)
	(set_tile Key Value)
)
;;; ------------ PROGRAM SUB-ROUTINES ------------
;;; ------------ SUB TO FIND A FILE AND RETURNS ITS PATH
(defun LC_FIND_FILE (DialogPrompt FileName Extension / Directory)

	(setq Directory (vl-filename-directory (findfile (getfiled DialogPrompt	FileName Extension 8))))
	Directory
)
;;; ------------ READ FILE SUB
(defun LC_READ_FILE (FileName / CurrentLine FileList LineNumber OpenFile)

	;; Open The file to read
	(setq OpenFile (open FileName "r"))
	;; Set line number counter
	(setq LineNumber 0)
	;; While there are lines to read
	(while (setq CurrentLine (read-line OpenFile))
		;; Check for empty lines
		(if (not (= "" CurrentLine))
			;; Add the current line the the list
			(setq FileList (cons (cons LineNumber CurrentLine) FileList))
		)
		;; Up the line number by one each line
		(setq LineNumber (1+ LineNumber))
	)
	;; Close the open file when done
	(close OpenFile)
	;; Reverse FileList to put it in order
	(setq FileList (reverse FileList))
	;; Send output
	FileList 
)
;;; ------------ REGISTRY CHECKING TO SEE IF KEY AND VALUE EXSIST
(defun LC_CHECK_REG (RegKey ValueName / RegCheck)

	;; Create directory structure
	(if (vl-registry-read RegKey ValueName); check for base directory
		(setq RegCheck T)
		(setq RegCheck nil)
	)
	RegCheck
)
;;; ------------ STRING TO LIST SUB ROUTINE, CREATE A LIST FROM A STRING WITH DELIMETER
(defun LC_STRING_TO_LIST (Stg Del / CurChr PosCnt TmpLst TmpStr NewTmpLst)

	(setq PosCnt 1)
	(setq TmpStr "")
	
	(repeat (1+ (strlen Stg))
		(setq CurChr (substr Stg PosCnt 1))
		(if (= CurChr Del)
			(progn
				(setq TmpLst (cons TmpStr TmpLst))
				(setq TmpStr "")
			)
			(setq TmpStr (strcat TmpStr CurChr))
		)
		(setq PosCnt
		       (1+ PosCnt))
	)
	(setq NewTmpLst (reverse TmpLst))
	NewTmpLst
)
;;; ------------ SUB TO RETURN MASS ASSOCIATION LIST
(defun LC_MASSOC (Key Alist)
	
	(cond
		((null Alist) '())
		(T
			(if (= (caar Alist) Key)
			 (cons (cdr (car Alist)) (LC_MASSOC Key (cdr Alist)))
			 (LC_MASSOC Key (cdr Alist))
			)
		)
	)
)
;;; ------------ CREATE A LIST OF LAYER GROUP FILES FOR POPUP LIST
(defun LC_GET_LAYER (Directory Extension / FileList FileListExt)

	;; Get list of files from directory w/ extension
	(setq FileListExt (vl-directory-files Directory (strcat "*" Extension) 1))
	;; Get list of files from directory w/o extension
	(setq FileList (mapcar '(lambda (x)(vl-filename-base x)) FileListExt))
	;; Send Output
	FileList
)
;;; ------------ Sub used to create layers
;;;
;;; Call it like this:
;;;		(LC_CREATE_LAYER "G-VPORT-NPLT" "Non-plotting viewport layer" "Continuous" "18" "12,12,12" "0" "Fine")
;;;
;;; (LC_CREATE_LAYER "NAME" "Descriptions" "Linetype" "Thickness" "Color" "Plot") - Plot is "1" NoPlot is "0"
(defun LC_CREATE_LAYER (Layer Descpition Linetype Thickness Color Plot Pstyle / VLA-Obj TmpList)

	;; Check color number
	(if (> (atoi Color) 255)
		(setq Color "7")
	)
	;; Check for truecolor********
	
	;;; ------------ CREATE A LIST FOR ENTMAKE
	(entmake
		(list
			(cons 0 "LAYER")
			(cons 100 "AcDbSymbolTableRecord")
			(cons 100 "AcDbLayerTableRecord")
			(cons 70 0)
			(cons 2 Layer)
			(cons 62 (atoi Color))
			(cons 6 Linetype)
			(cons 370 (atoi Thickness))
			(cons 290 (atoi Plot))
		)
	)
	;; Create layer description
	(if (>= (atof (getvar "acadver")) 16.1)
		(progn
			(setq VLA-Obj (vla-Add LayersCol	Layer))
			(vla-Put-Description VLA-Obj Descpition)
			(if (= (getvar "PSTYLEMODE") 0)		
				(vla-Put-PlotStyleName VLA-Obj Pstyle)
			)
			(vlax-release-object VLA-Obj)
		)
	)
)
;;; ------------ Insert Layers Sub
(defun LC_INSERT_LAYER (LayerLST / Item TmpLine LyrName LyrDescr LyrLine LyrWeight LyrColor LyrPlot 
	LyrPlotTable LyrPlotStyle LinePath LayerLST LayerCount VLA-Obj)

	;; Set counter for layer counting
	(setq LayerCount 0)
	; Set TmpLine to the listbox selection (item)
	(while (setq Item (read LayerLST))
		(setq LayerCount (1+ LayerCount))
		(setq TmpLine (LC_STRING_TO_LIST (cdr (assoc Item MasterList)) ";"))
		;; Set the layer properties
		(setq LyrName (nth 0 TmpLine))
		(setq LyrDescr (nth 1 TmpLine))
		(setq LyrLine (nth 2 TmpLine))
		(setq LyrWeight (nth 3 TmpLine))
		(setq LyrColor (nth 4 TmpLine))
		(setq LyrPlot (nth 5 TmpLine))
		(if (nth 6 TmpLine)
			(progn
				(setq LyrPlotTable (strcat (nth 6 TmpLine) ".stb"))
				(setq LyrPlotStyle (nth 7 TmpLine))
			)
		)	
		;; Load linetype if not already loaded
		(if (not (tblsearch "LTYPE" LyrLine))
			(if (setq LinePath (LC_FIND_LINETYPE LyrLine))
				(command "-linetype" "load" LyrLine LinePath "")
				;; Create the error linetype
				(progn
					(LC_CREATE_ERRORLINE)
					(setq LyrLine "ERROR")
				)					
			)
		)
		;;----------------------------------------------------------------------------
		(if (= 0 (getvar "PSTYLEMODE"))
			(progn
				;; Set the current Plotstyle Table current
				(if (not (= CurrentPstyle LyrPlotTable))
					(if (member LyrPlotTable PlotStyleTables)
						(vla-put-StyleSheet ActiveLayout LyrPlotTable)
					)
				)
				;; Add Plotstyle to dictionary so it can be accessed even is it doesn't exsist
				(if (not (member LyrPlotStyle (LC_MASSOC 3 (dictsearch (namedobjdict) "ACAD_PLOTSTYLENAME"))))
					(LC_ADD_PLOTSTYLE_DIC LyrPlotStyle)
				)
			)
		)
		
		;; Create the layer if not already loaded
		(if (not (tblsearch "LAYER" LyrName))
			(LC_CREATE_LAYER LyrName LyrDescr LyrLine LyrWeight LyrColor LyrPlot LyrPlotStyle)
			(if (= Gbl:Update "1")
				(progn
					;; Get the layer to update
					(setq VLA-Obj (vla-item LayersCol LyrName))
					;; Put the new properties in the layer
					(if (= "256" LyrColor);; Check for color 256 "bylayer"
						(setq LyrColor 7)
					)
					(vla-Put-Color VLA-Obj LyrColor)
					;; Put Linetype
					(vla-Put-Linetype VLA-Obj LyrLine)
					;; Put Linewieght
					(if (= "Default" LyrWeight)
						(vla-Put-Lineweight VLA-Obj -3)
						(vla-Put-Lineweight VLA-Obj LyrWeight)
					)
					;; Put Plottable
					(if (= LyrPlot "1")
						(vla-Put-Plottable VLA-Obj ':vlax-true)
						(vla-Put-Plottable VLA-Obj ':vlax-false)
					)
					;; Put Description
					(if (>= (atof (getvar "acadver")) 16.1)
						(vla-Put-Description VLA-Obj LyrDescr)
					)
					;; Put Plotstyle
					(if (= (getvar "PSTYLEMODE") 0)
						(vla-Put-PlotStyleName VLA-Obj LyrPlotStyle)
					)
					(vlax-release-object VLA-Obj)
				)
			)
		)
		;; Remove the first layer in the list and create the next one
		(setq LayerLST (substr LayerLST (+ 2 (strlen (itoa Item)))))
	)
	(if(= GBL:Current "1")
		(setvar "CLAYER" LyrName)
	)
	;; Processing prompt
	(princ (strcat "\n  " (rtos LayerCount 2 0) " Layers Processed... "))
	;; Reset system
	(LC_RESET_ENV)
)
;;; ------------ CREATE A LIST OF LAYER NAMES FROM THE LAYER GROUP FILE FOR EDIT BOX
(defun LC_CREATE_LAYER_NAMES (LyrNameList / TempLayerList)

	;; Create a list of just layer names from the layer group file
	(setq TempLayerList 
		(mapcar '(lambda (x)(nth 0 x)); ("A-SECT-MCUT" "A-SECT-PATT" "A-DETL-GRPH" "A-DETL-METR" "A-DETL-INPD")
			(mapcar '(lambda (x)(LC_STRING_TO_LIST x ";")); ("A-ROOF-RFDR" "Roof Drains" "Continuous" "25" "1")
				(mapcar '(lambda (x)(cdr x))LyrNameList); ("A-ROOF-RFDR;Roof Drains;Continuous;25;1;") - ("A-ROOF-RFDR;Roof Drains;Continuous;25;1;1;")
			)
		)
	)
	;; Send output
	TempLayerList
)
;;; ------------ SUB TO FIND A LINETYPE
(defun LC_FIND_LINETYPE (Linetype / LineFiles FullFile Found OpenFile CurrentLine Result)
	
	;; Check each search path for a .lin file
	(foreach Path (LC_STRING_TO_LIST (getenv "ACAD") ";")
		(if (setq LineFiles (vl-directory-files Path "*.lin"))
			(progn
				(foreach File LineFiles
					(setq FullFile (cons (strcat Path "\\" File) FullFile))
				)
				(setq Found (cons Path Found))
			)
		)
	)
	;; Read each line file found and check for the linetype
	(foreach LineFile FullFile
		(setq OpenFile (open LineFile "r"))
		(while (setq CurrentLine (read-line OpenFile))
			(if (wcmatch (strcase CurrentLine) (strcat "*" (strcase LineType) "*"))
				(setq Result Linefile)
			)
		)
		(close OpenFile)
	)
	;; Send result
	Result
)
;;; ------------ SUB TO CREATE AN ERROR LINETYPE IF DESIRED LINETYPE CAN NOT BE FOUND
(defun LC_CREATE_ERRORLINE (/ )
	
	;; Check for the style, if not there create it
	(if (not (tblsearch "LTYPE" "ERROR"))
		(entmake
			'(
				(0 . "LTYPE")
				(100 . "AcDbSymbolTableRecord")
				(100 . "AcDbLinetypeTableRecord")
				(2 . "ERROR")
				(3 . "LayerCreator Error ___ ___ ___ ___ ___")
				(70 . 0)
				(72 . 65)
				(73 . 2)
				(40 . 0.375)
				(49 . 0.25)
				(74 . 0)
				(49 . -0.125)
				(74 . 0)
			)
		)
	)
)
;;; ------------ CREATE AND INSERT ROMANS TEXT STYLE INTO DRAWING: USED FOR LINETYPES
(defun LC_CREATE_STYLE (/)

	;; Check for the style, if not there create it
	(if (not (tblsearch "style" "ROMANS"))
		(entmake
			'((0 . "STYLE")
					(100 . "AcDbSymbolTableRecord")
					(100 . "AcDbTextStyleTableRecord")
					(2 . "Romans")				; Style Name
					(70 . 0)								; Standard flag values
					(40 . 1.8)							; Fixed text height
					(41 . 1.0)							; Width Factor
					(50 . 0.0)							; Oblique angle
					(71 . 0)								; Text generation
					(42 . 2.0)							; Last Height Used
					(3 . "Romans.shx")		; Primary Font name
					(4 . "")								; Bigfont file name 0 for None
			)
		)
	)
)
;;; ------------ PLOT TABLE SUB-ROUTINE ------------
;;; Written by: Art Cooney
(defun LC_ADD_PLOTSTYLE_DIC  (PlotStyleName / NewPlotStyleDict PlotStyleDictInfo PlotStyleDictEname PlotStyleEname 
	XList PlaceHolder Ultimate PenUltimate TempDataList)
 
	;; make a list of entity definition data suitable for ENTMAKEX: 
	;; Start by getting any plotstyle ename from the existing dictionary: 
	(setq PlotStyleDictInfo (dictsearch (namedobjdict) "ACAD_PLOTSTYLENAME")) 
	(setq PlotStyleDictEname (cdr (assoc -1 PlotStyleDictInfo)))	
	;; first one found will do: 
	(setq PlotStyleEname (cdr (assoc 350 PlotStyleDictInfo)))	
	;; get the important stuff from the data list: 
	(setq XList (cdr (entget PlotStyleEname)))	
	;; entmakex a new placeholder (pstyle ename)
	(setq PlaceHolder (entmakex XList))	
	;; revise the dictionary info using the new placeholder ... 
	;; add (3 . "VAPOR") followed by (350 . <PlaceHolder>), 
	;; both placed before (100 . <>) and (340 . <>) 
	(setq Ultimate (last PlotStyleDictInfo)) 
	(setq PenUltimate (nth (- (length PlotStyleDictInfo) 2) PlotStyleDictInfo)) 
	(setq TempDataList (cdr (cdr (reverse PlotStyleDictInfo)))) 
	(setq NewPlotStyleDict (cons (cons 3 PlotStyleName) TempDataList)) 
	(setq NewPlotStyleDict (cons (cons 350 PlaceHolder) NewPlotStyleDict)) 
	(setq NewPlotStyleDict (cons PenUltimate  NewPlotStyleDict)) 
	(setq NewPlotStyleDict (cons Ultimate NewPlotStyleDict)) 
	(setq NewPlotStyleDict (reverse NewPlotStyleDict))  
	;; now (says Art) "use (entmod) on the dictionary you want to add it 
	;; to in order to let the dictionary know that it's the new owner." 
	(entmod NewPlotStyleDict)
	(princ)	
)
;;; ------------ LAYER EXPORTER SUB-ROUTINES ------------
;;; ------------ SUB TO EXPORT ALL OF THE LAYERS IN THE CURRENT DRAWING
(defun LC_LAYER_EXPORT (/ GroupName TableName LayerList)	
	
	;; Get layer group name
	(if (setq GroupName (LC_GET_GROUP))	
		(progn
			;; Get plot style table name
			(if (= OldPstyleMode 0)
				(setq TableName (LC_TABLE_NAME))
				(setq TableName "PlotStyle Table.ctb")
			)		
			;; Get the list of layers and there properties for the .lyr file - Sorted alpha
			(setq LayerList (acad_strlsort (LC_GET_LAYERLIST TableName)))
			;; Create the layer file
			(if (= T (LC_CREATE_LYRFILE GroupName LayerList))
				(princ (strcat "\n " (strcase GroupName) ".lyr file was created successfully....\n"))
				(princ (strcat "\n " (strcase GroupName) ".lyr file was NOT created successfully....\n"))
			)
			;; Reset Environment
			(LC_RESET_ENV)
			;; Restart LayerCreator
			(C:LAYER_CREATOR)
		)		
	)
	(princ)
)	
;;; ------------ LOAD A DIALOG TO GET LAYER GROUP NAME
(defun LC_GET_GROUP (/ Group_DCL End_Dialog GroupName)

	;; Set default group name
	(setq GroupName "<ENTER GROUP NAME>")	
	;; Start Dialog
	(setq Group_DCL (load_dialog "LAYER_CREATOR.dcl"))
	(if (not (new_dialog "LayerGroup" Group_DCL))
		(progn
			(alert "Layer_Creator.dcl not found\n\nPlace file in the search path \n		 then re-run the program.")
			(exit)
		)
	)
	;; Set layer group to default
	(set_tile "layer_group" Groupname)	
	;; Set focus to the Group Name
	(mode_tile "layer_group" 2)	
	;; Highlight tile contents
	(mode_tile "layer_group" 3)	
	;; Dim copyright
	(mode_tile "copyright" 1)	
	;; Action tiles
	(action_tile "layer_group" "(setq GroupName $value)")
	(action_tile "accept" "(done_dialog 1)")	
	(action_tile "cancel" "(done_dialog 0)")
	;; End dialog session
	(setq End_Dialog (start_dialog))
	;; Check exit method
	(cond
		((= End_Dialog 0)
			(princ "\n Layer Exporting terminated....No layer file created.")
			(setq GroupName nil)
			(unload_dialog Group_DCL)			
			(C:LAYER_CREATOR)
		)
		((= End_Dialog 1)
			(unload_dialog Group_DCL)
			(if (= GroupName "<ENTER GROUP NAME>")
				(progn
					(setq GroupName nil)
					(alert "No Group Name Defined")
					(LC_LAYER_EXPORT)
				)
			)	
		)
	)
	GroupName
)
;;; ------------ LOAD A DIALOG TO GET TABLE NAME
(defun LC_TABLE_NAME (/ TempList Table_DCL End_Dialog TableName)

	;; Set the default table name
	(setq TableName "LayerCreator.stb")	
	;; Sort Plotstyle table list with just .stb		
	(foreach Table PlotStyleTables
		(if (not (wcmatch Table "*.ctb"))
			(setq TempList (cons Table TempList))
		)
	)
	(setq PlotStyleTables TempList)	
	;; Start Dialog
	(setq Table_DCL (load_dialog "LAYER_CREATOR.dcl"))
	(if (not (new_dialog "LayerPlotTable" Table_DCL))
		(progn
			(alert "Layer_Creator.dcl not found\n\nPlace file in the search path \n		 then re-run the program.")
			(exit)
		)
	)
	;; Dim copyright
	(mode_tile "copyright" 1)	
	;; Load the listbox
	(start_list "table_name" 3)
	(mapcar 'add_list PlotStyleTables)
	(end_list)	
	;; Action tile statements
	(action_tile "table_name" "(setq TableName (nth (atoi $value) PlotStyleTables))")
	(action_tile "accept" "(done_dialog 1)")	
	;; End dialog session
	(setq End_Dialog (start_dialog))	
	;; Check exit method
	(cond
		((= End_Dialog 1)	
			(unload_dialog Table_DCL)
		)
	)
	TableName
)
;;; ------------ GET LIST OF LAYER AND PROPERTIES FOR .LYR FILE
(defun LC_GET_LAYERLIST (TableName / LayerName LayerColor LayerDesc LayerPlot LayerWeight LayerLinetype 
	LayerPlotStyle Line LineList)

	;; Loop through layer collection for information
	(vlax-for Layer LayersCol
		(setq LayerName (vla-get-name Layer))
		(setq LayerColor (rtos (vla-get-color Layer)2 0))
		(if (= "" (vla-get-description Layer))
			(setq LayerDesc (strcat "Layer Creator - " LayerName))
			(setq LayerDesc (vla-get-description Layer))
		)
		(if (= ':vlax-true (vla-get-plottable Layer))
			(setq LayerPlot "1")
			(setq LayerPlot "0")
		)
		(setq LayerWeight (rtos (vla-get-Lineweight Layer)2 0))
		(setq LayerLinetype (vla-get-Linetype Layer))
		(setq LayerPlotStyle (vla-get-PlotStylename Layer))
		(setq Line (strcat Layername ";" LayerDesc ";" LayerLinetype ";" LayerWeight ";" LayerColor ";" LayerPlot ";" (vl-filename-base TableName) ";" LayerPlotStyle ";"))

		(setq LineList (cons Line LineList))
		(setq LineList (reverse LineList))
	)
	LineList
)	
;;; ------------ SUB TO CREATE THE .LYR FILE
(defun LC_CREATE_LYRFILE (GroupName LayerList / File)
	
	;; Create the layer group file from the line list
	(setq File (open (strcat BaseDirectory "\\Layer Files\\" GroupName ".lyr") "w"));; use selected name and default layer group location
	(foreach Line LayerList	
		(write-line Line File)
	)
	(close File)
	T
)

;;; ------------ PROGRAM HELPER SUBROUTINES ------------
;;; ------------ SUB TO SET PROGRAM VARIALBES
(defun LC_SET_PROG (/ Temp LCDirectory)

	;; Check for Version
	(if 
		(and
			(LC_CHECK_REG "HKEY_CURRENT_USER\\Software\\Layer Creator" "Version"); Check that the registry key exsists
			(= "v1.9" (vl-registry-read "HKEY_CURRENT_USER\\Software\\Layer Creator" "Version")); Check that the version is correct
		)
		(setq Version (vl-registry-read "HKEY_CURRENT_USER\\Software\\Layer Creator" "Version")); Set the version variable
		(setq Version (vl-registry-write "HKEY_CURRENT_USER\\Software\\Layer Creator"	"Version" "v1.9")); Write Version to registry
	)
	;; Set directory path of Layer Creator.lsp file
	(if (null (findfile "Layer_Creator.lsp"))
		(progn
			(alert "LAYER_CREATOR.lsp was not found in the search path.")
			(setq LCDirectory (LC_FIND_FILE "Locating Layer Creator Directory" "Layer_Creator" "lsp"))
			(setq BaseDirectory (vl-registry-write "HKEY_CURRENT_USER\\Software\\Layer Creator" "BaseDirectory" LCDirectory))
		)
		(setq LCDirectory (vl-filename-directory (findfile "Layer_Creator.lsp")))
	)
	;; Check for BaseDirectory from registry
	(if (LC_CHECK_REG "HKEY_CURRENT_USER\\Software\\Layer Creator" "BaseDirectory")
		(setq BaseDirectory (vl-registry-read "HKEY_CURRENT_USER\\Software\\Layer Creator" "BaseDirectory"))
		(setq BaseDirectory (vl-registry-write "HKEY_CURRENT_USER\\Software\\Layer Creator" "BaseDirectory" (LC_FIND_FILE "Locating Layer Creator Directory" "Layer_Creator" "lsp")))
	)
	;; Check directories are the same
	(if (not (equal LCDirectory BaseDirectory))
		(setq BaseDirectory (vl-registry-write "HKEY_CURRENT_USER\\Software\\Layer Creator" "BaseDirectory" LCDirectory))
	)
	;; Check for LayerFiles folder
	(setq Temp (strcat BaseDirectory "\\Layer Files"))
	(if (not (vl-file-directory-p Temp))
		(progn
			(alert "** Layer Files folder was not found.  Add the Layer Files folder to the base directory and re-run **")
			(exit)
		)
	)	
	;; Create list of .lyr files from directory w/o extension
	(if (not (setq LayerGroupList (LC_GET_LAYER (strcat BaseDirectory "\\Layer Files") ".lyr")))
		(progn
			(alert "** No layer definition files were found **")
			(exit)
		)
	)
	;; If	GBL:Current does not exsist set it
	;;		Variable used to make inserted layer current
	(if (not GBL:Current)
		(setq GBL:Current "0"); Clear "Make Layer Current" toggle
	)
	;; If	GBL:Update does not exsist set it
	;;		Variable used to update inserted layers
	(if (not GBL:Update)
		(setq GBL:Update "1")
	)
	;; If LayerGroupPos does not exsist set it
	;;		Variable used to get position of Layer group Tile
	(if (not LayerGroupPos)
		(setq LayerGroupPos 0); Set LayerGroupPos variable to first group in list
	)
	;; If	LayerNamePos does not exsist set it
	;;		Variable used to get position of Layer name tile
	(if (not LayerNamePos)
		(setq LayerNamePos 0); Set LayerNamePos variable to first group in list
	)
)
;;; ------------ SHOW SLIDE SUB - SHOWS HORIZ, VERT SEPERATORS OR SLIDE
(defun LC_SHOW_SLIDE (Tile Slide Background / X Y)
	
	(setq X (dimx_tile Tile))
	(setq Y (dimy_tile Tile))
	(start_image Tile)
	(fill_image 0 0 X Y Background)
	(cond
		((= Slide "HSeperator.sld")
			(vector_image 0 0 X 0 8)
			(vector_image 1 1 X 1 7)
		)
		((= Slide "VSeperator.sld")
			(vector_image 1 1 1 Y 8)
			(vector_image 0 0 0 Y 7)
		)
		((slide_image 0 0 X Y (strcat "LAYER_CREATOR" " (" (vl-filename-base Slide) ")")))
	)
	(end_image)
)
;;; ------------ SET ENVIRONMENT SUB
(defun LC_SET_ENV (/ )

	;; Load VLISP functionality
	(vl-load-com)
	
	;; Set system variable
	(setq OldCmdecho (getvar "CMDECHO"))
	(setq OldClayer (getvar "CLAYER"))
	(setq OldExpert (getvar "EXPERT"))
	(setq OldPstyleMode (getvar "PSTYLEMODE"))

	(setvar "CMDECHO" 0)
	(setvar "EXPERT" 5)
	
	;; Set undo marker
	(command "_undo" "BEGIN")

	;; Set activeX variables
	(setq	ActiveDoc	(vla-get-activedocument (vlax-get-acad-object)))
	(setq Space
		(if (= (getvar "cvport") 1)
			(vla-get-paperspace ActiveDoc)
			(vla-get-modelspace ActiveDoc)
		)
	)
	(setq ActiveLayout (vla-get-ActiveLayout ActiveDoc))
	(setq LayersCol (vla-get-layers ActiveDoc))
	(setq CurrentLayout (vla-item (vla-get-layouts ActiveDoc)(getvar "ctab")))
	(setq CurrentPstyle (vla-get-stylesheet CurrentLayout))
	
	;; Get a list of availible Plotstyle tables
	;; Create layer description
	(if (>= (atof (getvar "acadver")) 16.1)
		(progn
			(setq PlotStyleTables
				(vlax-safearray->list 
					(vlax-variant-value 
						(vla-getplotstyletablenames 
							ActiveLayout
						)
					)
				)
			)
			(setq PlotStyleTables (list "" ""))
		)
	)
	;; If one is not assigned
	(if (= "" CurrentPstyle)
		(setq CurrentPstyle "Not Assigned")
	)	
)
;;; ------------ RESET ENVIRONMENT SETTINGS
(defun LC_RESET_ENV (/)

	;; Set undo marker
	(command "_undo" "END")	
	;; Release vlisp objects
	(vlax-release-object CurrentLayout)
	(vlax-release-object ActiveLayout)
	(vlax-release-object LayersCol)
	(vlax-release-object Space)
	(vlax-release-object ActiveDoc)		
	;; Set last layer current
	(if(= GBL:Current "0")
		(setvar "clayer" OldClayer)
	)	
	;; Reset system variables
	(setvar "expert" OldExpert)
	(setvar "cmdecho" OldCmdecho)	
	;; Quiet exit
	(princ)
)
;;; ------------ ABOUT ALERT BOX
(defun LC_ABOUT_LAYER (/ About_DCL End_Dialog)

	(setq About_DCL (load_dialog "LAYER_CREATOR.dcl"))
	(if (not (new_dialog "LayerCreatorAbout" About_DCL))
		(progn
			(alert "Layer_Creator.dcl not found\n\nPlace file in the search path \n		 then re-run the program.")
			(exit)
		)
	)
	(LC_SHOW_SLIDE "sep1" "HSeperator.sld" -15)
	(LC_SHOW_SLIDE "logo" "Logo.sld" -15)	
	(setq End_Dialog (start_dialog))
	(unload_dialog About_DCL)
)
;;; ------------ SUB TO RESET LAYER CREATOR...DELETES REGISTRY ENTRY
(defun C:LC_CLEAN (/)
	
	(vl-registry-delete "HKEY_CURRENT_USER\\Software\\Layer Creator")
	(princ "\n  Layer Creator has been reset ")
	(princ)
	(C:LAYER_CREATOR)
)
;;;
;;; ------------ Command Line Load Sequence--------------------------------------------
(princ "\nLayer Creator v1.9.8 \n Timothy Spangler, \n January, 2010....loaded.")
(print)
(princ "Type \"LC\" to start")
(print)

;;;;;;